#' # Reclassification of the HaeanCover data
#' doi:10.1594/PANGAEA.823677
#' Corresponding to Bumsuk Seo (Bumsuk.Seo@uni-bayreuth.de)

path.Pangaea <- "." # Set your Pangaea data directory here 
setwd(path.Pangaea)

lccs.filename <- "HaeanCover_Legend.xls" 
igbp.filename <- "HaeanCover_IGBP_lookup.xls"

require(rgdal)
require(xlsx)
require(RColorBrewer)
require(lattice)


plotting <- TRUE # To plot and save the figures. Plotting could take a few minutes.  
saving <- TRUE   # To save the reclassified data into ESRI Shape file format.

#` Load a custom function
source("Functions_LULC_Pangaea.R")

#` proj4strings
proj4.LL <- "+proj=longlat +ellps=WGS84 +datum=WGS84 +no_defs +towgs84=0,0,0" # WGS84 EPSG:4326 
proj4.UTM52N <- "+proj=utm +zone=52 +datum=WGS84 +units=m +no_defs +ellps=WGS84 +towgs84=0,0,0" # UTM52N (WGS84) EPSG:32652

#` Read the original data
HaeanCover <- readOGR(dsn=path.Pangaea, layer="HaeanCover", verbose=T, dropNULLGeometries=T, stringsAsFactors=FALSE)

#` Convert mixed LULC types to singular LULC types
preconversion.table <- read.xlsx(file=lccs.filename, sheetName = "correction")
colnames(preconversion.table) <- c("original.types", "replacement.types")
HaeanCover.pre <- reclassLULC(HaeanCover, rcl=preconversion.table, columns= c("LULC2009", "LULC2010", "LULC2011"))

haean.pre.levels <-sort( unique(c(HaeanCover.pre$LULC2009, HaeanCover.pre$LULC2010, HaeanCover.pre$LULC2011))) # Corrected after submission 
haean.pre.levels <- c(haean.pre.levels[haean.pre.levels!="NA"], "NA")

#` ## S2 Groups
conversion.table.all <- read.xlsx(file=lccs.filename, sheetIndex="types")
conversion.table <- conversion.table.all[,c("LULC.class", "Presence.of.Vegetation", "Edaphic.Condition", "Artificiality.of.Cover", "Life.form", "Crop.Type", "Life.cycle", "Group")]

conversion.table.S2 <- subset(conversion.table, select=c("LULC.class", "Group"))
colnames(conversion.table.S2) <- c("original.types", "replacement.types")

HaeanCover.S2 <- reclassLULC(HaeanCover.pre, rcl=conversion.table.S2, columns= c("LULC2009", "LULC2010", "LULC2011"))

HaeanCover.S2@data$LULC2009 <- factor(HaeanCover.S2@data$LULC2009)
HaeanCover.S2@data$LULC2010 <- factor(HaeanCover.S2@data$LULC2010)
HaeanCover.S2@data$LULC2011 <- factor(HaeanCover.S2@data$LULC2011)

if (saving) { 
    writeOGR(HaeanCover.S2, dsn=".", "HaeanCover_S2", driver="ESRI Shapefile", overwrite_layer = T, verbose = T)                               
}

if (plotting) { 
    
    S2.colors <- c(brewer.pal(n=8, name="Set1"),  "yellow", "grey70")
    
    p_S2.2009 <- spplot(HaeanCover.S2, "LULC2009", col.regions=S2.colors, lty=0, edge.col = "transparent", colorkey=F)
    p_S2.2010 <- spplot(HaeanCover.S2, "LULC2010", col.regions=S2.colors, lty=0, edge.col = "transparent", colorkey=F)
    p_S2.2011 <- spplot(HaeanCover.S2, "LULC2011", col.regions=S2.colors, lty=0, edge.col = "transparent", colorkey=F)
    
    S2.legend <- c(levels(HaeanCover.S2$LULC2009) , "Missing data")
    S2.legend2 <- unlist(lapply(X=strwrap(S2.legend, width=34, simplify=F), FUN= function(x) {paste(unlist(x), collapse=" \n ")}))
    
    cairo_pdf(filename="HaeanCover_S2.pdf", width=12, height=4, antialias="none")
    plot.new()
    trellis.par.set(axis.line=list(col=NA), )
    legend(x=0.8, y= 1.35, legend=S2.legend2, fill=c(S2.colors, "white"), cex=1, box.lty=0, xpd=NA, ncol=1, text.width=0.2, y.intersp=1.6)
    
    print(p_S2.2009, split=c(1, 1, 4, 1), newpage=FALSE) 
    print(p_S2.2010, split=c(2, 1, 4, 1), newpage=FALSE)
    print(p_S2.2011, split=c(3, 1, 4, 1), newpage=FALSE)
    
    text("(a)", x= -0.1, y=1.2, xpd=NA)
    text("(b)", x= 0.19, y=1.2,  xpd=NA)
    text("(c)", x= 0.49, y=1.2,  xpd=NA)
    
    dev.off()

}


#` ## FAO-LCCS 
conversion.table.LCCS <- subset(conversion.table, select=c("LULC.class", "Artificiality.of.Cover"))
colnames(conversion.table.LCCS) <- c("original.types", "replacement.types")

HaeanCover.LCCS <- reclassLULC(HaeanCover.pre, rcl=conversion.table.LCCS, columns= c("LULC2009", "LULC2010", "LULC2011"))

HaeanCover.LCCS@data$LULC2009 <- factor(HaeanCover.LCCS@data$LULC2009)
HaeanCover.LCCS@data$LULC2010 <- factor(HaeanCover.LCCS@data$LULC2010)
HaeanCover.LCCS@data$LULC2011 <- factor(HaeanCover.LCCS@data$LULC2011)

# [1] "Artificial Surfaces and Associated Area"
# [2] "Artificial Waterbodies, Snow and Ice"
# [3] "Bare Area"
# [4] "Cultivated and Managed Terrestrial Area"
# [5] "Cultivated Aquatic or Regularly Flooded Areas "
# [6] "Natural and Semi-Natural Aquatic or Regularly Flooded Vegetation"
# [7] "Natural and Semi-Natural Terrestrial Vegetation"
# [8] "Natural Waterbodies, Snow and Ice"


if (saving) { 
    writeOGR(HaeanCover.LCCS, dsn=".", "HaeanCover_LCCS", driver="ESRI Shapefile", overwrite_layer = T, verbose = T)          
    
}

if (plotting) { 
    lccs.colors <- c("pink", "aquamarine2", "brown", "darkorange", "greenyellow", "darkorchid", "forestgreen", "darkturquoise")
    
    p_LCCS.2009 <- spplot(HaeanCover.LCCS, "LULC2009", col.regions=lccs.colors, lty=0, edge.col = "transparent", colorkey=F)
    p_LCCS.2010 <- spplot(HaeanCover.LCCS, "LULC2010", col.regions=lccs.colors, lty=0, edge.col = "transparent", colorkey=F)
    p_LCCS.2011 <- spplot(HaeanCover.LCCS, "LULC2011", col.regions=lccs.colors, lty=0, edge.col = "transparent", colorkey=F)
    
    lccs.legend <- c(levels(HaeanCover.LCCS$LULC2009) , "Missing data")
    lccs.legend2 <- unlist(lapply(X=strwrap(lccs.legend, width=34, simplify=F), FUN= function(x) {paste(unlist(x), collapse=" \n ")}))
    
    cairo_pdf(filename="HaeanCover_LCCS.pdf", width=12, height=4, antialias="none") # takes a few minutes
    plot.new()
    trellis.par.set(axis.line=list(col=NA), )
    legend(x=0.77, y= 1.43, legend=lccs.legend2, fill=c(lccs.colors, "white"), cex=0.97, box.lty=0, xpd=NA, ncol=1, text.width=0.2, y.intersp=1.6)
    
    print(p_LCCS.2009, split=c(1, 1, 4, 1), newpage=FALSE) 
    print(p_LCCS.2010, split=c(2, 1, 4, 1), newpage=FALSE)
    print(p_LCCS.2011, split=c(3, 1, 4, 1), newpage=FALSE)
    
    text("(a)", x= -0.1, y=1.2, xpd=NA)
    text("(b)", x= 0.19, y=1.2,  xpd=NA)
    text("(c)", x= 0.49, y=1.2,  xpd=NA)
    
    dev.off()

}



#` ## IGBP 17 classes
conversion.table.IGBP <- read.xlsx(file=igbp.filename, sheetName="IGBPlookup")
colnames(conversion.table.IGBP) <- c("original.types", "replacement.types")

HaeanCover.IGBP <- reclassLULC(HaeanCover.pre, rcl=conversion.table.IGBP, columns= c("LULC2009", "LULC2010", "LULC2011"))

igbp.desc <- read.xlsx(file= igbp.filename, sheetName="IGBP_desc", stringAsFactors=F)

igbp.levels <- as.character(igbp.desc$IGBP.class)
igbp.labels <- igbp.desc$MODIS.12Q1.Class

HaeanCover.IGBP@data$LULC2009 <-  factor(HaeanCover.IGBP@data$LULC2009, levels=igbp.levels, labels= igbp.labels)
HaeanCover.IGBP@data$LULC2010 <-  factor(HaeanCover.IGBP@data$LULC2010, levels=igbp.levels, labels= igbp.labels)
HaeanCover.IGBP@data$LULC2011 <-  factor(HaeanCover.IGBP@data$LULC2011, levels=igbp.levels, labels= igbp.labels)

# MODIS12Q1Class    Code    Name
# 0     WAT    Water Bodies
# 1	    ENF	Evergreen Needleleaf Forests 
# 2	    EBF	Evergreen Broadleaf Forests 
# 3	    DNF	Deciduous Needleleaf Forests
# 4	    DBF	Deciduous Broadleaf Forests
# 5	    MF	Mixed Forests
# 6	    CSH	Closed Shrublands
# 7	    OSH	Open Shrublands
# 8	    WSA	Woody Savannas
# 9	    SAV	Savannas
# 10	GRA	Grasslands
# 11	WET	Permanent Wetlands
# 12	CRO	Croplands
# 13	URB	Urban and Built-Up Lands
# 14	CVM	Cropland/Natural Vegetation Mosaics
# 15    SNO	Snow and Ice
# 16	BRN	Barren or Sparsely Vegetated
# 99	INT	Interrupted Areas
# 100	MD

if (saving) { 
    writeOGR(HaeanCover.IGBP, dsn=".", "HaeanCover_IGBP", driver="ESRI Shapefile", overwrite_layer = T, verbose = T)                     
    
}

if (plotting) { 
    
    igbp.colors <- rgb(igbp.desc[,-c(1,2)]/255) # color map http://duckwater.bu.edu/lc/igbp.txt
    igbp.colors[16] <- "dimgrey" # Missing data
    igbp.colors[19] <- "white" # Missing data
   
    p2.haean.2009 <- spplot(HaeanCover.IGBP, "LULC2009", col.regions=igbp.colors, lty=0, edge.col = "transparent", colorkey=F)
    p2.haean.2010 <- spplot(HaeanCover.IGBP, "LULC2010", col.regions=igbp.colors, lty=0, edge.col = "transparent", colorkey=F)
    p2.haean.2011 <- spplot(HaeanCover.IGBP, "LULC2011", col.regions=igbp.colors, lty=0, edge.col = "transparent", colorkey=F)
    
    
    cairo_pdf(filename="HaeanCover_IGBP.pdf", width=12, height=4, antialias="none")
    
    plot.new()
    trellis.par.set(axis.line=list(col=NA))
    
    igbp.levels.leg <- igbp.levels
    igbp.levels.leg[15]<- "Cropland/Natural Vegetation\nMosaics"
    legend(x=0.85, y= 1.5, legend=igbp.levels.leg, fill=igbp.colors, cex=0.65, box.lty=0, xpd=NA)  
    
    text("(a)", x= -0.1, y=1.2, xpd=NA)
    text("(b)", x= 0.19, y=1.2,  xpd=NA)
    text("(c)", x= 0.49, y=1.2,  xpd=NA)
    
    print(p2.haean.2009, split=c(1, 1, 4, 1), newpage=FALSE)
    print(p2.haean.2010, split=c(2, 1, 4, 1), newpage=FALSE)
    print(p2.haean.2011, split=c(3, 1, 4, 1), newpage=FALSE)
    
    dev.off()
    
}
